/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: rcXPacket.h 12931 2019-04-04 10:06:45Z AlexanderMinor $
   Last Modification:
    $Author: AlexanderMinor $
    $Date: 2019-04-04 12:06:45 +0200 (Do, 04 Apr 2019) $
    $Revision: 12931 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         xx                   created
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file rcXPacket.h
/// Class implementing the CrcXpacket class interface
/////////////////////////////////////////////////////////////////////////////


/*****************************************************************************
****  INCLUDE FILES AND CONSTANT DEFINITIONS                              ****
*****************************************************************************/
#pragma once

#include "APIInterface.h"
#include "cifXErrors.h"
#include "Hil_SystemCmd.h"

#include <vector>

#include "TransportLayer.h"
#include "DeviceHandler.h"

#define CRCXPACKET_DEVICECNT               1
#define RCXPACKET_SYSTEMCHANNEL_INTERNAL   0

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CrcXPacket
/// Single instance object which handles the translation of the cifX API into
/// rcX packet objects
///////////////////////////////////////////////////////////////////////////////////////////
class CrcXPacket 
  : public CDataLayer
{
public:
  CrcXPacket(CTransportLayer* pcTransportLayer,
             CEndpoint*       pcEndpoint);
  virtual ~CrcXPacket(void);

  virtual bool            ReceiveData(PHIL_TRANSPORT_PACKET ptPacket);
  virtual void            SendComplete(PHIL_TRANSPORT_HEADER ptHeader);

  virtual int32_t         Init(void);
  virtual void            Deinit(void);
  
  virtual int32_t         EnumDevice( uint32_t ulBoard, 
                                      uint32_t ulSize, 
                                      void*         pvBoardInfo);

  virtual int32_t         EnumChannel(uint32_t ulBoard, 
                                      uint32_t ulChannel, 
                                      uint32_t ulSize, 
                                      void*         pvChannelInfo);

  virtual int32_t         RestartDevice(char* szBoard);

  ///////////////////////////////////////////
  // Not implemented netX API functions
  ///////////////////////////////////////////
  
  // xDriver functions handled implicitly during Init / Exit
  // 
  virtual int32_t xDriverOpen                 ( CIFXHANDLE* phDriver)
  {
    UNREFERENCED_PARAMETER( phDriver);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  
  virtual int32_t xDriverClose                ( CIFXHANDLE  hDriver)
  { 
    UNREFERENCED_PARAMETER( hDriver);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }

  virtual int32_t xDriverGetInformation       ( CIFXHANDLE  hDriver, uint32_t ulSize, void* pvDriverInfo)
  {
    UNREFERENCED_PARAMETER( hDriver);
    UNREFERENCED_PARAMETER( ulSize);
    UNREFERENCED_PARAMETER( pvDriverInfo);

    return CIFX_FUNCTION_NOT_AVAILABLE;     
  }

  virtual int32_t xDriverGetErrorDescription  ( int32_t lError, char* szBuffer, uint32_t ulBufferLen)
  {
    UNREFERENCED_PARAMETER( lError);
    UNREFERENCED_PARAMETER( szBuffer);
    UNREFERENCED_PARAMETER( ulBufferLen);

    return CIFX_FUNCTION_NOT_AVAILABLE; 
  }

  virtual int32_t xDriverEnumBoards           ( CIFXHANDLE  hDriver, uint32_t ulBoard, uint32_t ulSize, void* pvBoardInfo)
  {
    UNREFERENCED_PARAMETER( hDriver);
    UNREFERENCED_PARAMETER( ulBoard);
    UNREFERENCED_PARAMETER( ulSize);
    UNREFERENCED_PARAMETER( pvBoardInfo);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }

  virtual int32_t xDriverEnumChannels         ( CIFXHANDLE  hDriver, uint32_t ulBoard, uint32_t ulChannel, uint32_t ulSize, void* pvChannelInfo)
  {
    UNREFERENCED_PARAMETER( hDriver);
    UNREFERENCED_PARAMETER( ulBoard);
    UNREFERENCED_PARAMETER( ulChannel);
    UNREFERENCED_PARAMETER( ulSize);
    UNREFERENCED_PARAMETER( pvChannelInfo);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  
  virtual int32_t xDriverMemoryPointer        ( CIFXHANDLE  hDriver, uint32_t ulBoard, uint32_t ulCmd, void* pvMemoryInfo)
  {
    UNREFERENCED_PARAMETER( hDriver);
    UNREFERENCED_PARAMETER( ulBoard);
    UNREFERENCED_PARAMETER( ulCmd);
    UNREFERENCED_PARAMETER( pvMemoryInfo);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }


  virtual int32_t xSysdeviceOpen              ( CIFXHANDLE  hDriver, char*   szBoard, CIFXHANDLE* phSysdevice);
  virtual int32_t xSysdeviceClose             ( CIFXHANDLE  hSysdevice);
  virtual int32_t xSysdeviceGetMBXState       ( CIFXHANDLE  hSysdevice, uint32_t* pulRecvPktCount, uint32_t* pulSendPktCount); 
  virtual int32_t xSysdevicePutPacket         ( CIFXHANDLE  hSysdevice, CIFX_PACKET* ptSendPkt, uint32_t ulTimeout);
  virtual int32_t xSysdeviceGetPacket         ( CIFXHANDLE  hSysdevice, uint32_t ulSize, CIFX_PACKET* ptRecvPkt, uint32_t ulTimeout);
  virtual int32_t xSysdeviceInfo              ( CIFXHANDLE  hSysdevice, uint32_t ulCmd, uint32_t ulSize, void* pvInfo);

  virtual int32_t xSysdeviceFindFirstFile     ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xSysdeviceFindNextFile      ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xSysdeviceDownload          ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t ulFileSize, 
                                             PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xSysdeviceUpload            ( CIFXHANDLE  hSysdevice, uint32_t ulChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t* pulFileSize, 
                                             PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);

  virtual int32_t xSysdeviceReset             ( CIFXHANDLE  hSysdevice, uint32_t ulTimeout);
  virtual int32_t xSysdeviceResetEx           ( CIFXHANDLE  hSysdevice, uint32_t ulTimeout, uint32_t ulMode);

  virtual int32_t xChannelOpen                ( CIFXHANDLE  hDriver,  char* szBoard, uint32_t ulChannel, CIFXHANDLE* phChannel);
  virtual int32_t xChannelClose               ( CIFXHANDLE  hChannel);
  virtual int32_t xChannelFindFirstFile       ( CIFXHANDLE  hChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xChannelFindNextFile        ( CIFXHANDLE  hChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);

  virtual int32_t xChannelDownload            ( CIFXHANDLE  hChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t ulFileSize, 
                                             PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);
  virtual int32_t xChannelUpload              ( CIFXHANDLE  hChannel, uint32_t ulMode, char* szFileName, uint8_t* pabFileData, uint32_t* pulFileSize,
                                             PFN_PROGRESS_CALLBACK pfnCallback, PFN_RECV_PKT_CALLBACK pfnRecvPktCallback, void* pvUser);

  virtual int32_t xChannelGetMBXState         ( CIFXHANDLE  hChannel, uint32_t* pulRecvPktCount, uint32_t* pulSendPktCount);
  virtual int32_t xChannelPutPacket           ( CIFXHANDLE  hChannel, CIFX_PACKET*  ptSendPkt, uint32_t ulTimeout);
  virtual int32_t xChannelGetPacket           ( CIFXHANDLE  hChannel, uint32_t ulSize, CIFX_PACKET* ptRecvPkt, uint32_t ulTimeout);
  virtual int32_t xChannelGetSendPacket       ( CIFXHANDLE  hChannel, uint32_t ulSize, CIFX_PACKET* ptRecvPkt);

  virtual int32_t xChannelConfigLock          ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);
  virtual int32_t xChannelReset               ( CIFXHANDLE  hChannel, uint32_t ulResetMode, uint32_t ulTimeout);
  virtual int32_t xChannelInfo                ( CIFXHANDLE  hChannel, uint32_t ulSize, void* pvChannelInfo);
  virtual int32_t xChannelWatchdog            ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulTrigger);
  virtual int32_t xChannelHostState           ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);
  virtual int32_t xChannelBusState            ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout);

  virtual int32_t xChannelIOInfo              ( CIFXHANDLE  hChannel, uint32_t ulCmd,        uint32_t ulAreaNumber, uint32_t ulSize, void* pvData);
  virtual int32_t xChannelIORead              ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulOffset,     uint32_t ulDataLen, void* pvData, uint32_t ulTimeout);
  virtual int32_t xChannelIOWrite             ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulOffset,     uint32_t ulDataLen, void* pvData, uint32_t ulTimeout); 
  virtual int32_t xChannelIOReadSendData      ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulOffset,     uint32_t ulDataLen, void* pvData);

  virtual int32_t xChannelControlBlock        ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);
  virtual int32_t xChannelCommonStatusBlock   ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);
  virtual int32_t xChannelExtendedStatusBlock ( CIFXHANDLE  hChannel, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);
  virtual int32_t xChannelUserBlock           ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData);

  // xChannel function currently not supported because of missing packet commands
  //
  virtual int32_t xChannelPLCMemoryPtr        ( CIFXHANDLE  hChannel, uint32_t ulCmd,        void* pvMemoryInfo)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulCmd);
    UNREFERENCED_PARAMETER(pvMemoryInfo);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCIsReadReady      ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t* pulReadState)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);
    UNREFERENCED_PARAMETER(pulReadState);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCIsWriteReady     ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t* pulWriteState)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);
    UNREFERENCED_PARAMETER(pulWriteState);
    
    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCActivateWrite    ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }
  virtual int32_t xChannelPLCActivateRead     ( CIFXHANDLE  hChannel, uint32_t ulAreaNumber)
  {
    UNREFERENCED_PARAMETER(hChannel);
    UNREFERENCED_PARAMETER(ulAreaNumber);

    return CIFX_FUNCTION_NOT_AVAILABLE;
  }

protected:
  // Channel information structure                        
  typedef struct CHANNEL_INFO_Ttag
  {
    uint32_t                ulAreaNr;
    uint32_t                ulChannelId;
    HIL_FW_IDENTIFICATION_T tFirmwareInfo;

    HANDLE            hRecvSemaphore; /* Indicates stored packets in recv queue */
    HIL_PACKET_QUEUE  cvRecvQueue;

    CRITICAL_SECTION  tcsRecvLock;
    uint32_t          ulRecvCount;

    uint32_t          ulOpenCount;

    uint32_t          ulSendPacketCnt;
    uint32_t          ulRecvPacketCnt;
    uint32_t          ulHSKFlagSize;
    uint32_t          ulNumOfBlocks;
    uint32_t          ulPDInHskMode;
    uint32_t          ulPDOutHskMode;

  } CHANNEL_INFO_T;

  typedef std::map<BYTE, CHANNEL_INFO_T> CHANNEL_MAP;

  CEndpoint*        m_pcEndpoint;

  LONG              m_lRefCount;          //!< Reference count of this object

  // Member functions
  int32_t ReadHardwareIdentify   ( HIL_HW_IDENTIFY_CNF_DATA_T* ptHardwareIdentify);           //!< Read device information
  int32_t ReadHardwareInfo       ( HIL_HW_HARDWARE_INFO_CNF_DATA_T* ptHardwareInfo);          //!< Read HW Information
  int32_t ReadSysInformation     ( SYSTEM_CHANNEL_SYSTEM_INFORMATION* ptSystemInfo);     //!< Read system information 
  int32_t ReadSysInfoBlock       ( HIL_DPM_SYSTEM_INFO_BLOCK_T* ptSysInfoBlock); //!< Read system info block
  int32_t ReadSysControlBlock    ( HIL_DPM_SYSTEM_CONTROL_BLOCK_T* ptSysControlBlock);  //!< Read system control block
  int32_t ReadSysStatusBlock     ( HIL_DPM_SYSTEM_STATUS_BLOCK_T* ptSysStatusBlock);    //!< Read system status block

  int32_t ReadChannelInfo        ( uint32_t ulAreaIndex, CHANNEL_INFO_T* ptChannelInfo); //!< Read channel information
  int32_t ReadBlockInfo          ( uint32_t ulAreaIndex, uint32_t ulSubblockIdx, HIL_DPM_GET_BLOCK_INFO_CNF_DATA_T* ptBlockInfo);
  int32_t ReadComFlags           ( uint32_t ulAreaIndex, uint32_t* pulNetxComFlag, uint32_t* pulHostComFlag);
  int32_t ReadChannelFirmware    ( uint32_t ulChannel, HIL_FW_IDENTIFICATION_T* ptFirmware);
  int32_t ReadChannelInfoBlock   ( uint32_t ulChannel, uint32_t ulSize, void* pvData);  //!< Read system channel block
  int32_t ReadChannelControlBlock( uint32_t ulChannel, HIL_DPM_CONTROL_BLOCK_T* ptControlBlock);
  int32_t ReadCommonStatusBlock  ( uint32_t ulChannel, HIL_DPM_COMMON_STATUS_BLOCK_T* ptStatusBlock);

  int32_t Reset                  ( HANDLE hDevice, uint32_t ulResetMode, uint32_t ulTimeout);
  int32_t ReadDPMIOInfo          ( HANDLE hDevice, CHANNEL_INFORMATION* ptChannelInfo);

  uint32_t        BuildHandle(BYTE bBoard, 
                              BYTE bChannel);

  bool            InterpretHandle(uint32_t ulHandle, 
                                  BYTE&         bBoard, 
                                  BYTE&         bChannel);

  void            GenerateBoardInformation(BOARD_INFORMATION* ptBoardInfo);
  void            GenerateChannelInfo(CHANNEL_INFO_T& tChannelInfo, CHANNEL_INFORMATION* ptChannelInfo);

  int32_t         GetChannelId(uint8_t bChannel, uint32_t& ulChannelId)
  {
    int32_t lRet = CIFX_NO_ERROR;

    EnterCriticalSection(&m_tcsChannels);

    CHANNEL_MAP::iterator iter = m_cmChannels.find(bChannel);

    if(iter == m_cmChannels.end())
    {
      lRet = CIFX_INVALID_HANDLE;
    } else
    {
      ulChannelId = iter->second.ulChannelId;
    }

    LeaveCriticalSection(&m_tcsChannels);

    return lRet;
  }

  CHANNEL_MAP                           m_cmChannels;         //!< Channel map (HilTransport.bChannel->ChannelData)
  CRITICAL_SECTION                      m_tcsChannels;        //!< Critical Section for the mvChannels vector                               
  
  uint32_t                              m_ulSendPktRefCnt;    //!< Reference counter of outstanding transfers
  uint32_t                              m_ulDeviceCnt;        //!< Device counter                                           
  std::string                           m_szDeviceName;       //!< Device name 

  HANDLE                                m_hReConnectEvent;    /*!< Event, which should be signaled on reconnect */

  HIL_HW_IDENTIFY_CNF_DATA_T            m_tDeviceInfo;        //!< Hardware configuration data
  HIL_DPM_SYSTEM_INFO_BLOCK_T           m_tSystemInfo;        //!< System information of the device
};
